---
marp: true
paginate: true
math: mathjax
---

# Rustbelt, a formalization of Rust type system

### Ralf Jung et al. @ POPL'18

### Presented by Yanning Chen @ ProSE Seminar

---

# Features of Rust type system

- Ownership
- Mutable/Shared References
- Lifetimes
- Interior Mutability

**Goal**: Well-typed Rust programs should be *memory-safe*.

**How?** *Aliasing* and *mutation* cannot ocur at the same time on any given location.

---

# Ownership

In Rust, a type represents:

+ information on what values it can hold
+ **ownership of the resources (e.g. memory)**

**Fact**: Rust uses an affine type system, i.e. a value can be used *at most* once.
**Consequence**: no two values can own the same resource, so *mutation* allowed but **no** *aliasing*.

```rust
let s1 = String::from("hello");
let s2 = s1; // s1 is moved to s2, i.e. s1 is used and no longer available
println!("{}", s1); // error: use of moved value: s1
```

*Q: What happens when a value is weakened?*
A: Underlying resources deallocated!

---

# Mutable Reference

What we don't want to gain permanent access to a value?

```rust
fn Vec::push<T>(Vec<T>, T) -> Vec<T>

let v_ = Vec::push(v, 1);   // v is no longer available
```

Instead, Rust uses *mutable references*:

```rust
fn Vec::push<T>(&mut Vec<T>, T)

Vec::push(&mut v, 1);   // v is still available
```

A **mutable reference** grants *temporary exclusive access* (i.e. *borrowing*) to the value  *for the duration of the function call*. 
**Result**: still, *mutation* allowed but **no** *aliasing*.

---

# Shared Reference

*What if we want to access a value at multiple places?*
Admit *aliasing*!

```rust
let v = vec![1];
// Two shared references to v are created
join(|| println!("{:?}", &v), || println!("{:?}", &v));
// Still have access to v at the main thread after references are dropped
Vec::push(&mut v, 2);
```

**Result**: for memory safety, allow *aliasing* but **no** *mutation*.

```rust
let v = vec![1];
let r = &v;             // temporary shared reference
Vec::push(&mut v, 2);   // error: active shared reference exists
println!("{:?}", r);    // shared reference ends here
```

---

# Shared Reference - *Copy* types

*What if we want to access a value at multiple places?*
Admit *aliasing*!

Therefore, *shared references* can be freely duplicated, i.e. *unrestricted variables* in linear logic.

In Rust, *unrestricted types* are called *Copy* types.

Semantically, every type that can be duplicated via bit-wise copy is a *Copy* type.

- `&T` yes, because it's a shared pointer. `Int` yes, because it's a number.

- `&mut T` no, because it also holds exclusive access. `Vec<Int>` no, because it's a pointer to an heap array, and a bit-wise copy doesn't duplicate the underlying data.

---

# Lifetimes

- **Ownership**: exclusive access, *mutation*
- **Mutable Reference**: *temporary* exclusive access, *mutation*
- **Shared Reference**: *temporary* shared access, *aliasing*

How to track if a reference is active? How long is *temporary*?
Answer: equip each reference with a *lifetime*.

```rust
&'a mut T   // mutable reference with lifetime 'a
&'b T       // shared reference with lifetime 'b
```

---

# Lifetimes

![w:800](index_sig.png)
![w:1000](lifetime_example.png)

- the output of `index_mut` has the same lifetime as the input.
- passing `v` to `index_mut`, we create a lifetime `'b` for `v` and `head`.
- to call `push` we need to create a mutable reference, whose lifetime overlaps with `'b`.

---

# Interior Mutability

**Q**: What if we need shared mutable state? i.e. multi-thread queue?
**A**: Add primitives that allow *mutation* through *shared references*, i.e. *interior mutability*.

```rust
Cell::set(&Cell<T>, T)
Cell::get(&Cell<T>) -> T
```

```rust
let c1 : &Cell<i32> = &Cell::new(1);
let c2 : &Cell<i32> = &c1;
c1.set(2);
println!("{}", c2.get()); // 2
```

**Oops!** *Aliasing* and *mutation* at the same time!

`Cell` is implemented using **unsafe** code, i.e. opting *out* of the type system.

---

# Interior Mutability

If you think about it, `Cell` is still safe to use.

```rust
Cell::set(&Cell<T>, T)
Cell::get(&Cell<T>) -> T
```

`Cell` can only hold *Copy* types, and returns a copy of the value when `get` is called.

No way to alias the inner data semantically!

---

# Formalization of Rust: Challenges

+ Complex language: imperative, traits, ...
+ *Unsafe* types: opting out of syntactic typing rules

---

# Challenge: complex language

**Solution**: work on a subset of Rust intermediate representation called $\lambda_{Rust}$.

![w: auto](lrust_skim.png)

---

# Type system of $\lambda_{Rust}$

**Observation**: local variables of Rust are also addressable.

**Simplification**: treat local variables as heap-allocated, i.e. *pointer* types.

- Primitives: `bool`, `int`
- Pointers: 
  1. $\textbf{own}\ \tau$: pointer with full ownership of an allocation containing a value of type $\tau$
  2. $\&_{\textbf{mut}/\textbf{shr}}^\kappa \tau$: mutable/shared reference with lifetime $\kappa$ to a value of type $\tau$
- Other types: $\Pi$, $\Sigma$, $\to$, $\mu$, ...

*Note*: Types of local variables of Rust programs are all *pointer* types.

*Not describing in detail due to time limit.*

---

# Challenge: *unsafe* types

*Unsafe* types opts out of typing rules, so no way to prove safety from the rules!

**Solution**: take the *semantic* approach.

- **syntactic typing**: terms the typing rules allow to be of type $\tau$
- **semantic typing**: terms that are safe to be treated as type $\tau$

---

# Semantic typing

*What is a type?* a certain set of values, or, a predicate on values.

**Example**: in lambda calculus with booleans,

- $[|\text{Bool}|](v) := v = \text{true} \lor v = \text{false}$.
- $[|A * B|](v) := \exists v_1, v_2. v = (v_1, v_2) \land [|A|](v_1) \land [|B|](v_2)$.

---

# Challenges to model Rust type system

+ How to describe *ownership*?
+ How to describe *temporary* access?
+ How to deal with *interior mutability*?

---

# Challenge: How to describe *ownership*?

*What is a type?* a certain set of values, or, a predicate on values.

*What predicate? using which logic?*

**Separation Logic!**

---

# Separation Logic 101

A logic that describes a *heap*.

$\def\wand{\mathrel{-\mkern-6mu*}}$

- $\texttt{emp}$: empty heap
- $x \mapsto v$: heap with a single cell at address $x$ containing value $v$
- $P * Q$: heap that can be *split* into two parts, one satisfying $P$ and the other satisfying $Q$ (like *conjunction*, but disjoint)
- $P \wand Q$: heap that, *disjointly* combined with another heap satisfying $P$, satisfies $Q$ (like *implication*, but disjoint)

---

# Separation Logic 101

Separation logic is a substructural logic.

**Example**: Consider the following heap: $x = 1$.
$x \mapsto 1$ holds, but $x \mapsto 1 * x \mapsto 1$ does not. Thus, no *contraction*.

Also, after an implication is applied to a value, the value is *consumed*.

---

# Separation Logic 101

A logic that describes a *heap*.

Separation logic is a substructural logic.

- Rust types: a type represents ownership of a resource, and the type system is affine.
- Separation logic: a predicate represents a resource, and the logic is *affine*.

Perfect logic to describe Rust types!

*P.S. 🤓👆 Not every separation logic is affine, but the one used in Rustbelt, i.e. Iris,  is.*

---

# Interpreting Rust types: primitives

Associate every type $\tau$ to an *Iris* (separation logic) predicate on values.

$[|\tau|].\text{own}: \textit{list Val} \to Prop$

*(Ignore why we name it "own" for now, will be explained later.)*

- $[|\textbf{bool}|].\text{own}(\bar{v}) := \bar{v} = [\textbf{true}] \lor \bar{v} = [\textbf{false}]$
- $[|\tau_1 \times \tau_2|].\text{own}(\bar{v}) := \exists \bar{v_1}, \bar{v_2}. \bar{v} = \bar{v_1} \texttt{++} \bar{v_2} * [|\tau_1|].\text{own}(\bar{v_1}) * [|\tau_2|].\text{own}(\bar{v_2})$

**Notice**: $*$ is *separating conjunction*, meaning its two oprands are disjoint in memory.

---

# Interpreting Rust types: *Copy* types

**Recall**: types that can be freely duplicated via bit-wise copy are *Copy* types.
**Consequence**: given $[|\tau|].\text{own}(\bar{v})$, we can freely duplicate the proposition, recovering contraction rule on the type.

*Proposition that can be freely copied (i.e. $P \vdash P * P$) is called a persistent proposition.*

Therefore, the interpretation of *Copy* types can always be written as:

$[|\tau|].\text{own}(\bar{v}) := \exists v. \bar{v} = [v]. * \Phi_{\tau}(v)$, where $\Phi_{\tau}$ is a persistent proposition.

E.g. for $\tau = \textbf{bool}$, $\Phi_{\textbf{bool}}(v) := v = [\textbf{true}] \lor v = [\textbf{false}]$, which is trivially persistent because it's not describing any resource.

---

# Interpreting Rust types: owned pointers

Associate every type $\tau$ to an *Iris* (separation logic) predicate on values.

$[|\textbf{own}\ \tau|].\text{own}(\bar{v}) := \exists \mathscr{l}. \bar{v} = [\mathscr{l}] * \exists \bar{w}. \mathscr{l} \mapsto \bar{w} * [|\tau|].\text{own}(\bar{w})$

- $\exists \mathscr{l}. \bar{v} = [\mathscr{l}]$: $\bar{v}$ contains a single address $\mathscr{l}$.
- $\mathscr{l} \mapsto \bar{w}$: heap at address $\mathscr{l}$ contains value $\bar{w}$.
- $[|\tau|].\text{own}(\bar{w})$: $\bar{w}$ can be seen as a value of type $\tau$.

**Notice**: $*$ is *separating conjunction*, meaning location $\mathscr{l}$ and memory region representing $\bar{w}$ are disjoint.

---

# † Interpreting Rust types: owned pointers (for *Copy* types)

$[|\textbf{own}\ \tau|].\text{own}(\bar{v}) := \exists \mathscr{l}. \bar{v} = [\mathscr{l}] * \exists \bar{w}. \mathscr{l} \mapsto \bar{w} * [|\tau|].\text{own}(\bar{w})$

**Recall**: types that can be duplicated via bit-wise copy are *Copy* types.

Try to duplicate $[|\textbf{own}\ \tau|].\text{own}(\bar{v})$:

- $\exists \mathscr{l'}. \bar{v} = [\mathscr{l'}]$: can always find another address $\mathscr{l}'$. (assume no allocation failure)
- $\exists \bar{w'}. \mathscr{l'} \mapsto \bar{w'}$: let $\bar{w'} = \bar{w}$ up to bit-wise copy.
- $[|\tau|].\text{own}(\bar{w'})$: holds because $\tau$ can be duplicated by bit-wise copy.

**Property**: for any *Copy* type $\tau$, predicate $[|\textbf{own}\ \tau|](\bar{v})$ can be freely duplicated.

---

# Interpreting Rust types: mutable references

What's the difference between *mutable references* and *owned pointers*?

- *owned pointers*: ownership for an unlimited time
- *mutable references*: ownership for *a limited period of time*

---

# Challenge: how to describe *temporary* ownership?

Recall how we tracked references in Rust type system: *lifetimes*.

**Solution**: lifetime logic.

## Full borrow predicate

$P$: separation assertion representing ownership of some resource
$\&_{\textbf{full}}^\kappa P$: assertion representing ownership of $P$ *during lifetime $\kappa$*

**Intuition**: $P$ holds only when $\kappa$ is active.

*We'll head back to the precise definition of lifetime logic later.*

---

# Interpreting Rust types: mutable references

$\&_{\textbf{mut}}^\kappa \tau$: mutable reference with lifetime $\kappa$ to a value of type $\tau$

**Meaning**: ownership of a value of type $\tau$ for the duration of lifetime $\kappa$.

- $[|\textbf{own}\ \tau|].\text{own}(\bar{v}) := \exists \mathscr{l}. \bar{v} = [\mathscr{l}] * \exists \bar{w}. \mathscr{l} \mapsto \bar{w} * [|\tau|].\text{own}(\bar{w})$

- $[|\&_{\textbf{mut}}^\kappa \tau|].\text{own}(\bar{v}) := \exists \mathscr{l}. \bar{v} = [\mathscr{l}] * \&_{\textbf{full}}^{\kappa}( \exists \bar{w}. \mathscr{l} \mapsto \bar{w} * [|\tau|].\text{own}(\bar{w}))$

---

# Interpreting Rust types: shared references

$[|\&_{\textbf{shr}}^\kappa \tau|].\text{own}(\bar{v}) := ?$

**Question**: what can we say about shared references *universally*?

1. they are pointers
1. they are *Copy* types, i.e. can be freely duplicated
1. they can be created by downgrading a *mutable reference*
1. for *Copy* $\tau$, we can bit-wise copy the value it points to and get a new $\tau$

*Not so interesting!* Is that true?

**Interior mutability**!

---

# How to deal with *interior mutability*?

Many types have their own sharing reference behavior deviating from the universal rules!

**Solution**: let every type define their own sharing reference behavior, i.e. *sharing predicate*.

- **owned predicate** $[|\tau|].\text{own}(\bar{v})$: describe values $\bar{v}$ that can be considered as type $\tau$
- **sharing predicate** $[|\tau|].\text{shr}(\kappa, \mathscr{l})$: describe a location $\mathscr{l}$ and lifetime $\kappa$ to be considered as type $\&_{\textbf{shr}}^\kappa \tau$

Leveraging the sharing predicate to describe the behavior of shared references.

$[|\&_{\textbf{shr}}^\kappa \tau|].\text{own}(\bar{v}) := \exists \mathscr{l}. \bar{v} = [\mathscr{l}] * [|\tau|].\text{shr}(\kappa, \mathscr{l})$

---

# Interpreting Rust types: shared references

Leveraging the sharing predicate to describe the behavior of shared references.

$[|\&_{\textbf{shr}}^\kappa \tau|].\text{own}(\bar{v}) := \exists \mathscr{l}. \bar{v} = [\mathscr{l}] * [|\tau|].\text{shr}(\kappa, \mathscr{l})$

Laws for sharing predicates:

1. ~~they are pointers~~: already satisfied by the definition of sharing predicate
1. they ~~are *Copy* types~~ can be freely duplicated: $[|\tau|].\text{shr}(\kappa, \mathscr{l})$ must be persistent.
1. they can be created by downgrading a *mutable reference*: $[|\&_{\textbf{mut}}^\kappa \tau|].\text{own}([\mathscr{l}]) * [\kappa]_q \ \wand \ [|\tau|].\text{shr}(\kappa, \mathscr{l}) * [\kappa]_q$

*$[\kappa]_q$ is a token that asserts the lifetime $\kappa$ is active, and we'll talk about it later.*

---

# Interpreting Rust types: shared references

4. for *Copy* $\tau$, we can bit-wise copy the value it points to and get a new $\tau$.

**Recall**: for *Copy* types $\tau$,

$[|\tau|].\text{own}(\bar{v}) := \exists v. \bar{v} = [v]. * \Phi_{\tau}(v)$, where $\Phi_{\tau}$ is a persistent proposition.

**Define**:
$[|\tau|].\text{shr}(\kappa, \mathscr{l}) := \exists v. \&_{frac}^{\kappa}(\mathscr{l} \mapsto^q v) * \Phi_{\tau}(v)$

---

# Interpreting Rust types: shared references

**Define**: for *Copy* types $\tau$,
$[|\tau|].\text{shr}(\kappa, \mathscr{l}) := \exists v. \&_{frac}^{\kappa}(\mathscr{l} \mapsto^q v) * \Phi_{\tau}(v)$

**Recall**: for mutable references,
$[|\&_{\textbf{mut}}^\kappa \tau|].\text{own}(\bar{v}) := \exists \mathscr{l}. \bar{v} = [\mathscr{l}] * \&_{\textbf{full}}^{\kappa}( \exists \bar{w}. \mathscr{l} \mapsto \bar{w} * [|\tau|].\text{own}(\bar{w}))$

**Intuition**: † *fractured borrow* $\&_{frac}^{\kappa} P$ also represents ownership $P$ during lifetime $\kappa$, but:

- is *persistent*, because it represents a shared borrow, while full borrow is not
- only grants a fraction of its content ($\mapsto^q$)

*†: no need to understand the details. Just treat them as **full borrow**s.*

---

# Lifetime logic

Things we used but not defined yet:

- **Full borrow** $\&_{\textbf{full}}^\kappa P$: assertion representing ownership of $P$ *during lifetime $\kappa$*
- † **Fractured borrow** $\&_{frac}^\kappa P$: assertion representing ownership of $P$ *during lifetime $\kappa$*, but only grants a fraction of its content
- **Lifetime token** $[\kappa]_q$: token that asserts the lifetime $\kappa$ is active

---

# Lifetime logic

```rust
let mut v = Vec::new();
v.push(0);
{ // <- Vec<i32>
  let mut head = v.index_mut(0);
  *head = 23;
}
println!("{:?}", v);
```

given that

```rust
index_mut: for<'a> fn(&'a mut Vec<i32>, usize) -> &'a mut i32
```

---

# Lifetime logic

```rust
index_mut: for<'a> fn(&'a mut Vec<i32>, usize) -> &'a mut i32
```

```rust
{ 
  let mut head = v.index_mut(0); // <- Vec<i32>
```

- need to provide `'a`
- need to pass a mutable reference of lifetime `'a`

---

# Lifetime logic

```rust
index_mut: for<'a> fn(&'a mut Vec<i32>, usize) -> &'a mut i32
```

```rust
{ 
  let mut head = v.index_mut(0); // <- Vec<i32> * [κ] * ([κ] -* [†κ])
```

- need to provide `'a`

  ${\rm L{\small FT}L}\texttt{-}\rm{\small BEGIN}$: $\texttt{True} \ \wand\  \exists \kappa. [\kappa]_1 * ([\kappa]_1 \wand [\dagger \kappa])$

  Can always

  - create a lifetime token $[\kappa]_1$, accompanied by
  - a way to end it $[\kappa]_1 \wand [\dagger \kappa]$. ($[\dagger \kappa]$ is a token that asserts the lifetime $\kappa$ has ended.)

---

# Lifetime logic

```rust
index_mut: for<'a> fn(&'a mut Vec<i32>, usize) -> &'a mut i32
```

```rust
{ 
  let mut head = v.index_mut(0); // <- &κ mut Vec<i32> * [κ] * ([†κ] -* Vec<i32>) * ([κ] -* [†κ])
```

- need to provide `'a` (done)
- need to pass a mutable reference of lifetime `'a`

  $\rm L{\small FT}L\texttt{-}\rm{\small BORROW}$: $P \ \wand\  \&_{\textbf{full}}^{\kappa} P * ([\dagger \kappa] \wand P)$

  Given an owned resource $P$, can split it into
  - a full borrow $\&_{\textbf{full}}^{\kappa} P$, and
  - an *inheritance* $[\dagger \kappa] \wand P$ that can retrieve $P$ back after $\kappa$ dies.

---

# Lifetime logic: separating conjunction

${\rm L{\small FT}L}\texttt{-}\rm{\small BEGIN}$: $\texttt{True} \ \wand\  \exists \kappa. [\kappa]_1 * ([\kappa]_1 \wand [\dagger \kappa])$
$\rm L{\small FT}L\texttt{-}\rm{\small BORROW}$: $P \ \wand\  \&_{\textbf{full}}^{\kappa} P * ([\dagger \kappa] \wand P)$

- Sep logic $P * Q$: *heap* that can be *split* into two *disjoint* parts, one satisfying $P$ and the other satisfying $Q$
- Lifetime logic $P * Q$: *time* that can be *split* into two *disjoint* parts, one satisfying $P$ (when $\kappa$ is alive) and the other satisfying $Q$ (when $\kappa$ is dead)

---

# Lifetime logic: frame rule

It's important for $P$ and $Q$ to be *disjoint*.

Consider $P \land Q$ and $P * Q$.

$
\cfrac{P \vdash P' \quad Q \vdash Q'}{P * Q \vdash P' * Q'}
\quad \text{(can be seen as)}\ 
\cfrac{\forall x, \{P(x)\}\ c\ \{P'(x)\} \quad \forall x, \{Q(x)\}\ c\ \{Q'(x)\}}{\forall x, \{(P * Q)(x)\}\ c\ \{(P' * Q')(x)\}}
$

But for $P \land Q$,

$
\cfrac{\forall x, \{P(x)\}\ c\ \{P'(x)\} \quad \forall x, \{Q(x)\}\ c\ \{Q'(x)\}}{\forall x, \{P(x) \land Q(x)\}\ c\ \{?\}}
$

What if $P$ and $Q$ describes some shared resource, and while $P ⊢ P'$, $c$ modifies something that invalidates $Q$?

---

# Lifetime logic: separating conjunction

${\rm L{\small FT}L}\texttt{-}\rm{\small BEGIN}$: $\texttt{True} \ \wand\  \exists \kappa. [\kappa]_1 * ([\kappa]_1 \wand [\dagger \kappa])$
$\rm L{\small FT}L\texttt{-}\rm{\small BORROW}$: $P \ \wand\  \&_{\textbf{full}}^{\kappa} P * ([\dagger \kappa] \wand P)$

Whatever we do about $\&_{\textbf{full}}^{\kappa} P$, we can always get back the *inheritance*.

---

# Lifetime logic

```rust
index_mut: for<'a> fn(&'a mut Vec<i32>, usize) -> &'a mut i32
```

```rust
{ 
  let mut head = v.index_mut(0); // <- inside `index_mut`
```

1. split input `&κ Vec<i32>` into the accessed `&κ i32` and the rest `&κ Vec<i32>`
2. return `&k i32` to the caller, and drop the rest

---

# Lifetime logic

```rust
index_mut: for<'a> fn(&'a mut Vec<i32>, usize) -> &'a mut i32
```

```rust
{ 
  let mut head = v.index_mut(0); // <- inside `index_mut`
```

1. split input `&κ Vec<i32>` into the accessed `&κ i32` and the rest `&κ Vec<i32>`
  $\rm L{\small FT}L\texttt{-}\rm{\small BOR}\texttt{-}\rm{\small SPLIT}$: $\&_{\textbf{full}}^{\kappa} (P * Q) \vdash \&_{\textbf{full}}^{\kappa} P * \&_{\textbf{full}}^{\kappa} Q$
2. return `&κ i32` to the caller, and drop the rest
  $P * Q \vdash P$, because *Iris* is an affine logic

```rust
{ 
  let mut head = v.index_mut(0); // <- &κ mut i32 * [κ] * ([†κ] -* Vec<i32>) * ([κ] -* [†κ])
```

---

# Lifetime logic

```rust
  let mut head = v.index_mut(0);
  *head = 23;   // <- i32 * (i32 -* &κ mut i32 * [κ]) * ([†κ] -* Vec<i32>) * ([κ] -* [†κ])
```

Need to access the resource of mutable reference `head`.

$\rm L{\small FT}L\texttt{-}\rm{\small BOR}\texttt{-}\rm{\small ACC}$: $\&_{\textbf{full}}^{\kappa} P * [\kappa]_q \ \wand\  P * (P \wand \&_{\textbf{full}}^{\kappa} P * [\kappa]_q)$

Given a full borrow $\&_{\textbf{full}}^{\kappa} P$ and a witness $[\kappa]_q$ that shows $\kappa$ is active,

- can access the resource $P$, accompanied by
- an *inheritance* $P \wand \&_{\textbf{full}}^{\kappa} P * [\kappa]_q$ that can retrieve mutable reference and *lifetime token back* after the access

*It's important to return things you borrowed!*: *lifetime token* is such a certificate.

---

# Lifetime logic

```rust
  *head = 23;   // <- &κ mut i32 * [κ] * ([†κ] -* Vec<i32>) * ([κ] -* [†κ])
}
```

```rust
  *head = 23;
}               // <- [κ] * ([†κ] -* Vec<i32>) * ([κ] -* [†κ])
```

```rust
  *head = 23;
}               // <- ([†κ] -* Vec<i32>) * [†κ]
```

```rust
  *head = 23;
}               // <- Vec<i32>
```

---

# † Lifetime logic

**Fractured borrow** $\&_{frac}^{\kappa}$ vs **Full borrow** $\&_{\textbf{full}}^{\kappa}$

- **Fractured borrow**s are persistent: can be accessed simultaneously by multiple parties (freely duplicatable), but do not have full access, i.e. only a fraction of the resource.
- It's always possible to take a little bit of a resource from a **Fractured borrow**, no matter how many times it's been borrowed.

**Intuition**:
- from a full borrow with full lifetime $[\kappa]_1$, by downgrading it to a fractured borrow, we can get a fraction of it, thus getting fractional lifetime $[\kappa]_q$, e.g. $[\kappa]_{0.1}$, which is shorter than $[\kappa]_1$.
- The semantics guarantees that we can always get a tiny bit of resource of lifetime $[\kappa]_\epsilon$ from a fractured borrow.

---

# Proof of soundness

Typing judgments are defined as

$\textbf{L} | \textbf{T} \vdash \textrm{I} \dashv x.\textbf{T}'$

- $\textbf{L}$ lifetime context
- $\textbf{T}$ type context
- $\textrm{I}$ instruction

After the instruction, the type context is updated to $\textbf{T}'$ with new variable $x$ added.

---

# Proof of soundness

Interpretation of typing judgments:

$\textbf{L} | \textbf{T} \vDash \textrm{I} \ \ \mathbin{\style{display: inline-block; transform: scaleX(-1); padding: -4px}{\vDash}} x.\textbf{T}' :=
\{[|\textbf{L}|]_\gamma * [|\textbf{T}|]_\gamma\}\ \textrm{I}\ \{\exists v. [|\textbf{L}|]_\gamma * [|\textbf{T}'|]_{\gamma[x \leftarrow v]}\}$

- Interpreted as a separation logic triple
- $[|\textbf{T}|]$ uses interpretation of types described earlier

---

# Proof of soundness

1. **FTLR** (Foundamental Theorem of Logical Relations):
  $\forall \textbf{L}, \textbf{T}, \textrm{I}. \quad \textbf{T}'. \textbf{L} | \textbf{T} \vdash \textrm{I} \dashv x.\textbf{T}' \Rightarrow \textbf{L} | \textbf{T} \vDash \textrm{I} \ \ \mathbin{\style{display: inline-block; transform: scaleX(-1); padding: -4px}{\vDash}} x.\textbf{T}'$
  *Syntactic typing rules are sound w.r.t. semantic typing rules.*

2. **Adequacy**: a semantically well-typed program never gets stuck (no invalid memory access or data race).

**Collary**: every rust program that consists of *syntactically* well-typed *safe* code and *semantically* well-typed *unsafe* code, is safe to execute.

---

# Conclusion

- Rust type system: ownership, mutable/shared references, lifetime, interior mutability
- Formalization: $\lambda_{Rust}$, $\textbf{own}\ \tau$, $\&_{\textbf{mut/shr}}^\kappa$. Unsafe types? *Semantic typing*!
- Semantic typing:
  - Separation logic
  - $[| \tau |].\text{own}(\bar{v})$, $[| \tau |].\text{shr}(\kappa, \mathscr{l})$ (for interior mutability)
  - $\&_{\textbf{full}}^\kappa P$, $\&_{frac}^\kappa P$, $[\kappa]_q$? *Lifetime logic*!
- Lifetime logic by example
  - Fractured borrow: persistent + fractional (inclusion) lifetime
- Soundness proof:
  - Judgment interpreted as separation logic triple
  - FTLR (syntactic -> semantic) + Adequacy (semantic -> runtime)

---

# Appendix: Lifetime logic meets Interior Mutability

**Example**: Mutex is a product of flag (true: locked, false: unlocked) and the resource.

$[|\textbf{mutex}(\tau)|].\text{own}(\bar{v}) := [| \textbf{bool} \times \tau |].\text{own}(\bar{v})$

$[|\textbf{mutex}(\tau)|].\text{shr}(κ, \mathscr{l}) := \&_{\textbf{atom}}^\kappa($
$\quad \mathscr{l} \mapsto \textbf{true} \ \lor$
$\quad \mathscr{l} \mapsto \textbf{false} * \&_{\textbf{full}}^{\kappa}(\exists \bar{v}. (\mathscr{l}+1) \mapsto \bar{v} * [|\tau|].\text{own}(\bar{v}))$
$)$

**Atomic persistent borrow** $\&_{\textbf{atom}}^\kappa P$: assertion representing ownership of $P$ that *cannot be accessed for longer than one single instruction cycle*. Can be freely duplicated.

---

# Appendix: Lifetime logic meets Interior Mutability

**Example**: Mutex is a product of flag (true: locked, false: unlocked) and the resource.

$[|\textbf{mutex}(\tau)|].\text{shr}(κ, \mathscr{l}) := \&_{\textbf{atom}}^\kappa($
$\quad \mathscr{l} \mapsto \textbf{true} \ \lor$
$\quad \mathscr{l} \mapsto \textbf{false} * \&_{\textbf{full}}^{\kappa}(\exists \bar{v}. (\mathscr{l}+1) \mapsto \bar{v} * [|\tau|].\text{own}(\bar{v}))$
$)$

**Atomic persistent borrow** $\&_{\textbf{atom}}^\kappa P$: assertion representing ownership of $P$ that *cannot be accessed for longer than one single instruction cycle*. Can be freely duplicated.

- When unlocked, one thread borrows it, takes its inner full borrow away, and set lock flag. Other threads can't observe an intermediate state due to atomicity.
- Later, another thread tries to borrow it, but the lock flag is set.
- When the first thread releases the lock, it put back the full borrow so another thread can use it.